--══════════════════════════════════════════════════════════════════════════════
--  hybrid_flow_strip.lua   • 500 × 375 Scriptable Display
--══════════════════════════════════════════════════════════════════════════════
--  1  Boost bar   : kersInput  – body = BOOST_BODY, tip = BOOST_TIP
--  2  Fuel bar    : fuel / maxFuel – body + tip
--  3  ERS-left    : kersCharge – body + tip
--  4  Charging    : wipe while kersCharge rises – body + tip
--  Debug preview  : DEBUG=true  or hold Extra C
--══════════════════════════════════════════════════════════════════════════════

------------------------------------- USER-CONFIG -----------------------------
local TIP_STYLE   = "fade"   -- "solid" | "fade"
local TIP_W       = 450      -- width of tip / fade zone (px)
local DEBUG       = false    -- permanent preview? (true/false)
local FILL_SPEED  = 1250     -- px/s wipe speed for charging bar

-- BAR POSITIONS
local BOOST = { x = 3,  y =  42, w = 501, h = 20 }
local FUEL  = { x = 3,  y = 104.5, w = 501, h = 21 }
local ERS   = { x = 4,  y = 2194.0, w = 493, h = 20 }
local CHRG  = { x = 25, y = 2246.5, w = 449, h = 25 }

-- COLOURS
local BOOST_BODY = rgb(0.00,0.35,0.55); local BOOST_TIP = rgb(1,1,1)
local FUEL_BODY  = rgb(1,1,1)         ; local FUEL_TIP  = rgb(1,1,1)
local ERS_BODY   = rgb(1,1,1); local ERS_TIP   = rgb(1,1,1)
local CHRG_BODY  = rgb(0,0.35,0.55)         ; local CHRG_TIP  = rgb(1,1,1)

-- DATA FIELDS
local FIELD_INPUT   = 'kersInput'
local FIELD_BATTERY = 'kersCharge'
-------------------------------------------------------------------------------

local BG  = rgb(1,1,1,0)
local OUT = rgbm(1,1,1,0)

-- helpers --------------------------------------------------------------------
local function clamp(v,a,b) return (v<a) and a or (v>b) and b or v end
local function safe(o,k) local ok,v=pcall(function()return o[k] end) return ok and v or 0 end
local function mix(a,b,t) return rgb(
  a.r+(b.r-a.r)*t, a.g+(b.g-a.g)*t, a.b+(b.b-a.b)*t) end

local function rect(x,y,w,h,c) display.rect{pos=vec2(x,y),size=vec2(w,h),color=c} end
local function outline(b)
  rect(b.x,   b.y-1, b.w,1, OUT); rect(b.x, b.y+b.h, b.w,1, OUT)
  rect(b.x-1, b.y-1,1,b.h+2, OUT); rect(b.x+b.w,b.y-1,1,b.h+2, OUT)
end

-- draw body + tip (solid or fade) -------------------------------------------
local function drawTwoColorBar(b, frac, bodyCol, tipCol)
  rect(b.x,b.y,b.w,b.h,BG)
  if frac>0 then
    local fillW = math.min(b.w, math.floor(b.w*frac+0.5))
    if TIP_STYLE == "solid" then
      local bodyW = math.max(fillW - TIP_W, 0)
      if bodyW>0 then rect(b.x,b.y,bodyW,b.h,bodyCol) end
      rect(b.x+bodyW,b.y,fillW-bodyW,b.h,tipCol)
    else                                   -- fade style
      local fadeW = math.min(fillW, TIP_W)
      local bodyW = fillW - fadeW
      if bodyW>0 then rect(b.x,b.y,bodyW,b.h,bodyCol) end
      local slices = 30
      local segW   = fadeW / slices
      for i=0,slices-1 do
        local x  = b.x + bodyW + math.floor(i*segW+0.5)
        local w  = math.ceil(segW)
        if x + w > b.x + fillW then w = b.x + fillW - x end
        if w<=0 then break end
        rect(x,b.y,w,b.h,mix(bodyCol,tipCol,(i+0.5)/slices))
      end
    end
  end
  outline(b)
end
-------------------------------------------------------------------------------

local car        = ac.getCar(0)
local lastCharge = safe(car,FIELD_BATTERY)
local fillPx     = 0

function script.update(dt)
  if not car then return end
  local debugOn = DEBUG

  -- 1 BOOST (now uses BOOST_BODY / BOOST_TIP) --------------------------------
  local boostFrac = debugOn and 1 or clamp(safe(car,FIELD_INPUT),0,1)
  drawTwoColorBar(BOOST, boostFrac, BOOST_BODY, BOOST_TIP)

  -- 2 FUEL -------------------------------------------------------------------
  local fuelFrac  = debugOn and 1 or clamp(car.fuel / math.max(car.maxFuel,1),0,1)
  drawTwoColorBar(FUEL,  fuelFrac,  FUEL_BODY, FUEL_TIP)

  -- 3 ERS SOC ---------------------------------------------------------------
  local socFrac   = debugOn and 1 or clamp(safe(car,FIELD_BATTERY),0,1)
  drawTwoColorBar(ERS,   socFrac,   ERS_BODY,  ERS_TIP)

  -- 4 CHARGING wipe ---------------------------------------------------------
  local batt      = safe(car,FIELD_BATTERY)
  local charging  = debugOn or (batt > lastCharge)
  lastCharge      = batt

  local targetPx  = charging and CHRG.w or 0
  if fillPx < targetPx then
    fillPx = math.min(fillPx + FILL_SPEED*dt, targetPx)
  elseif fillPx > targetPx then
    fillPx = math.max(fillPx - FILL_SPEED*dt, targetPx)
  end

  drawTwoColorBar(CHRG, fillPx/CHRG.w, CHRG_BODY, CHRG_TIP)
end
